/***
 * CopyPolicy: GNU Public License 2 applies
 * Copyright (C) 1994-1997 Heiko Eissfeldt heiko@colossus.escape.de
 *
 * Interface module for cdrom drive access
 *
 * Two interfaces are possible.
 *
 * 1. using 'cooked' ioctls() (Linux only)
 *    : available for atapi, sbpcd and cdu31a drives only.
 *
 * 2. using the generic scsi device (for details see SCSI Prog. HOWTO).
 *    NOTE: a bug/misfeature in the kernel requires blocking signal
 *          SIGINT during SCSI command handling. Once this flaw has
 *          been removed, the sigprocmask SIG_BLOCK and SIG_UNBLOCK calls
 *          should removed, thus saving context switches.
 *
 * For testing purposes I have added a third simulation interface.
 *
 * Version 0.8: used experiences of Jochen Karrer.
 *              SparcLinux port fixes
 *              AlphaLinux port fixes
 *
 */
#if 0
#define SIM_CD
#endif

#include "config.h"
#include <standard.h>
#include <stdio.h>
#include <stdlib.h>
#if defined (HAVE_UNISTD_H) && (HAVE_UNISTD_H == 1)
#include <sys/types.h>
#include <unistd.h>
#endif
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <fcntl.h>
#include <assert.h>

#include <sys/ioctl.h>
#include <sys/stat.h>
#include <statdefs.h>


#include "mycdrom.h"
#include "lowlevel.h"
/* some include file locations have changed with newer kernels */
#if defined (__linux__)
# if LINUX_VERSION_CODE > 0x10300 + 97
#  if LINUX_VERSION_CODE < 0x200ff
#   include <linux/sbpcd.h>
#   include <linux/ucdrom.h>
#  endif
#  if !defined(CDROM_SELECT_SPEED)
#   include <linux/ucdrom.h>
#  endif
# endif

/* Hack hack hack hack hack... makes the code clean, though. Zygo was here */
# ifndef SG_BIG_BUFF
#  define SG_BIG_BUFF 4096	/* FIXME: should be the size of page */
# endif
/* HACK! the kernel header files were misdesigned for
 *  user applications.
 * #include <linux/blkdev.h> did not work
 */
struct request {		/* this is and may not used */
  int dummy;
};
#endif

#include "byteorder.h"
#include "interface.h"
#include "cdda2wav.h"
#include "semshm.h"
#include "setuid.h"
#include "ringbuff.h"
#include "toc.h"
#include "global.h"
#include "scsi_cmds.h"

#include <utypes.h>
#include <scsitransp.h>
#include <cdrecord.h>

unsigned interface;

int trackindex_disp = 0;

TOC g_toc [MAXTRK]; /* 100 */
unsigned char MCN[14];  /* including a zero string terminator */

void     (*EnableCdda) __PR((int Switch));
unsigned (*ReadToc) __PR(( TOC *ptoc ));
unsigned (*ReadLastAudio) __PR(( unsigned tracks ));
void     (*ReadCdRom) __PR(( UINT4 *p, unsigned lSector, unsigned SectorBurstVal ));
void     (*ReadCdRomData) __PR(( unsigned char *p, unsigned lSector, unsigned SectorBurstVal ));
subq_chnl *(*ReadSubQ) __PR(( unsigned char sq_format, unsigned char track ));
void     (*SelectSpeed) __PR(( unsigned speed ));
int	(*Play_at) __PR(( unsigned int from_sector, unsigned int sectors));
void	(*trash_cache) __PR((UINT4 *p, unsigned lSector, unsigned SectorBurstVal));


static int	Is_a_Toshiba3401;

/* hook */
static void Dummy __PR(( void ));
static void Dummy ( )
{
}

#if !defined(SIM_CD)


LOCAL int
scsi_scandev __PR(( char *devp, int *xp1, int *xp2, int *xp3));

/*
 * Convert scgp->target,lun or scsibus,target,lun syntax.
 * Check for bad syntax and invalid values.
 * This is definitely better than using scanf() as it checks for syntax errors.
 */
LOCAL int
scsi_scandev(devp, xp1, xp2, xp3)
	char	*devp;
	int	*xp1;
	int	*xp2;
	int	*xp3;
{
	int	n = 0;

	*xp1 = *xp2 = *xp3 = 0;

	if (*devp != '\0') {
		devp = astoi(devp, xp1);
		if (*devp == ',') {
			devp++;
			n++;
		} else {
			return (-1);
		}
	}
	if (*devp != '\0') {
		devp = astoi(devp, xp2);
		if (*devp == ',' || *devp == '\0') {
			if (*devp != '\0')
				devp++;
			n++;
		} else {
			return (-1);
		}
	}
	if (*devp != '\0') {
		devp = astoi(devp, xp3);
		if (*devp == '\0') {
			n++;
		} else {
			return (-1);
		}
	}
	if (*xp1 < 0 || *xp2 < 0 || *xp3 < 0)
		return (-1);
	return (n);
}

int Toshiba3401 __PR((void));

int Toshiba3401 ( ) {
  return Is_a_Toshiba3401;
}

static void trash_cache_SCSI __PR((UINT4 *p, unsigned lSector, unsigned SectorBurstVal));

static void trash_cache_SCSI(p, lSector, SectorBurstVal)
	UINT4 *p;
	unsigned lSector;
	unsigned SectorBurstVal;
{
      /* trash the cache */
      ReadCdRom( p, find_an_off_sector(lSector, SectorBurstVal), min(global.nsectors,6));
}


static void Check_interface_for_device __PR((struct stat *statstruct, char *pdev_name));
static int OpenCdRom __PR((char *pdev_name));

static void SetupSCSI __PR((void));

static void SetupSCSI( )
{
    unsigned char *p;

    if (interface != GENERIC_SCSI) {
	/* unfortunately we have the wrong interface and are
	 * not able to change on the fly */
	fprintf(stderr, "The generic SCSI interface and devices are required\n");
	exit(1);
    }

    /* do a test unit ready to 'init' the device. */
    TestForMedium();

    /* check for the correct type of unit. */
    p = Inquiry();

#undef TYPE_ROM
#define TYPE_ROM 5
#undef TYPE_WORM
#define TYPE_WORM  4
    if (p == NULL || (*p != TYPE_ROM && *p != TYPE_WORM)) {
	fprintf(stderr, "this is neither a scsi cdrom nor a worm device\n");
	exit(1);
    }

    /* generic Sony type defaults */
    density = 0x0;
    accepts_fua_bit = -1;
    accepts_fua_bit = -1;
    EnableCdda = (void (*) __PR((int)))Dummy;
    ReadCdRom = ReadCdda12;
    ReadCdRomData = (void (*) __PR((unsigned char *, unsigned, unsigned ))) ReadStandard;
    ReadLastAudio = ReadFirstSessionTOCSony;
    SelectSpeed = SpeedSelectSCSISony;
    Play_at = Play_atSCSI;
    trash_cache = trash_cache_SCSI;

    /* check for brands and adjust special peculiaritites */

    /* If your drive is not treated correctly, you can adjust some things
       here:

       global.littleendian: should be to 1, if the CDROM drive or CD-Writer
		  delivers the samples in the native byteorder of the audio cd
		  (LSB first).
		  HP CD-Writers need it set to 0.
       NOTE: If you get correct wav files when using sox with the '-x' option,
             the endianess is wrong. You can use the -C option to specify
	     the value of global.littleendian.

     */

    {
      int mmc_code;

      allow_atapi(1);
      if (*p == TYPE_ROM) {
        mmc_code = heiko_mmc();
      } else {
        mmc_code = 0;
      }
      switch (mmc_code) {
       case 2:      /* SCSI-3 cdrom drive with accurate audio stream */
         if ( global.littleendian == -1 )
	   global.littleendian = 1;
         global.overlap = 0;
         ReadCdRom = ReadCddaMMC12;
         ReadLastAudio = ReadFirstSessionTOCMMC;
         SelectSpeed = SpeedSelectSCSIMMC;
       break;
       case 1:      /* SCSI-3 cdrom drive with no accurate audio stream */
         if ( global.littleendian == -1 )
	   global.littleendian = 1;
         global.overlap = 1;
         ReadCdRom = ReadCddaMMC12;
         ReadLastAudio = ReadFirstSessionTOCMMC;
         SelectSpeed = SpeedSelectSCSIMMC;
         break;
       case -1: /* "MMC drive does not support cdda reading, sorry\n." */
	/* fall through */
       case 0:      /* non SCSI-3 cdrom drive */
         ReadLastAudio = NULL;
    if (!memcmp(p+8,"TOSHIBA", 7) ||
        !memcmp(p+8,"IBM", 3) ||
        !memcmp(p+8,"DEC", 3)) {
	density = 0x82;
	EnableCdda = EnableCddaModeSelect;
 	ReadCdRom = ReadStandard;
        SelectSpeed = SpeedSelectSCSIToshiba;
        if (!memcmp(p+16, "CD-ROM XM-3401",14)) {
	   Is_a_Toshiba3401 = 1;
	}
	if ( global.littleendian == -1 )
		global.littleendian = 1;
    } else if (!memcmp(p+8,"IMS",3) ||
               !memcmp(p+8,"KODAK",5) ||
               !memcmp(p+8,"RICOH",5) ||
               !memcmp(p+8,"HP",2) ||
               !memcmp(p+8,"PHILIPS",7) ||
               !memcmp(p+8,"PLASMON",7) ||
               !memcmp(p+8,"GRUNDIG CDR100IPW",17) ||
               !memcmp(p+8,"MITSUMI CD-R ",13)) {
	EnableCdda = EnableCddaModeSelect;
	ReadCdRom = ReadStandard;
        SelectSpeed = SpeedSelectSCSIPhilipsCDD2600;

	/* treat all of these as bigendian */
	if ( global.littleendian == -1 )
		global.littleendian = 0;

	/* no overlap reading for cd-writers */
	global.overlap = 0;
    } else if (!memcmp(p+8,"NRC",3)) {
        SelectSpeed = NULL;
    } else if (!memcmp(p+8,"YAMAHA",6)) {
	EnableCdda = EnableCddaModeSelect;
        SelectSpeed = SpeedSelectSCSIYamaha;

	/* no overlap reading for cd-writers */
	global.overlap = 0;
	if ( global.littleendian == -1 )
		global.littleendian = 1;
    } else if (!memcmp(p+8,"PLEXTOR",7) ||
               !memcmp(p+8,"SONY",4)) {
	if ( global.littleendian == -1 )
		global.littleendian = 1;
        if (!memcmp(p+16, "CD-ROM CDU55E",13)) {
	   ReadCdRom = ReadCddaMMC12;
	}
        ReadLastAudio = ReadFirstSessionTOCSony;
    } else if (!memcmp(p+8,"NEC",3)) {
	ReadCdRom = ReadCdda10;
        SelectSpeed = SpeedSelectSCSINEC;
	if ( global.littleendian == -1 )
		global.littleendian = 1;
        if (!memcmp(p+29,"5022.0r",3)) /* I assume all versions of the 502 require this? */
               global.overlap = 0;           /* no overlap reading for NEC CD-ROM 502 */
    } else if (!memcmp(p+8,"MATSHITA",8)) {
	ReadCdRom = ReadCdda12Matsushita;
	if ( global.littleendian == -1 )
		global.littleendian = 1;
    }
    } /* switch (get_mmc) */
    }

    ReadToc = (!memcmp(p+8, "IMS     CDD5",12)) ? ReadTocCdrSCSI : ReadTocSCSI;
    ReadSubQ = ReadSubQSCSI;

    /* look if caddy is loaded */
    while (!TestForMedium()) {
	fprintf(stderr,"load cdrom please and press enter");
	getchar();
    }
}




EXPORT int
open_scsi __PR((char *scsidev, int os_debug, int be_verbose));
/*
 * Open a SCSI device.
 *
 * Possible syntax is:
 *
 * Preferred:
 *	dev=target,lun / dev=scsibus,target,lun
 *
 * Needed on some systems:
 *	dev=devicename:target,lun / dev=devicename:scsibus,target,lun
 *
 * On systems that don't support SCSI Bus scanning this syntax helps:
 *	dev=devicename:@ / dev=devicename:@,lun
 * or	dev=devicename (undocumented)
 *
 * NOTE: As the 'lun' is part of the SCSI command descriptor block, it
 * 	 must always be known. If the OS cannot map it, it must be
 *	 specified on command line.
 */
EXPORT int
open_scsi(scsidev, os_debug, be_verbose)
	char	*scsidev;
	int	os_debug;
	int	be_verbose;
{
	char	devname[256];
	char	*devp = NULL;
	int	x1, x2, x3;
	int	n = 0;

	devname[0] = '\0';
	if (scsidev != NULL) {
		if ((devp = strchr(scsidev, ':')) == NULL) {
			if (strchr(scsidev, ',') == NULL) {
				/* Notation form: 'devname' (undocumented)   */
				/* Fetch bus/tgt/lun values from OS	     */
				n = -1;
				global.lun = -2;	/* Lun must be known	     */
				strncpy(devname, scsidev, sizeof(devname)-1);
				devname[sizeof(devname)-1] = '\0';
			} else {
				/* Basic notation form: 'bus,tgt,lun'	     */
				devp = scsidev;
			}
		} else {
			/* Notation form: 'devname:bus,tgt,lun'/'devname:@'  */
			x1 = devp - scsidev;
			if (x1 >= (int)sizeof(devname))
				x1 = sizeof(devname)-1;
			strncpy(devname, scsidev, x1);
			devname[x1] = '\0';
			devp++;
			/* Check for a notation in the form 'devname:@'	     */
			if (devp[0] == '@') {
				if (devp[1] == '\0') {
					global.lun = -2;
				} else if (devp[1] == ',') {
					if (*astoi(&devp[2], &global.lun) != '\0')
						return -1;
				}
				n = -1;
				devp = NULL;
			}
		}
	}
	if (devp != NULL) {
		n = scsi_scandev(devp, &x1, &x2, &x3);
		if (n < 0) {
			errno = EINVAL;
			return -1;
		}
	}
	if (n == 3) {		/* Got bus,target,lun			*/
		scsibus = x1;
		target = x2;
		lun = x3;
	} else if (n == 2) {	/* Got global.target,lun			*/
		scsibus = 0;
		target = x1;
		lun = x2;
	} else if (n == -1) {	/* Got device:@, fetch bus/lun from OS	*/
		scsibus = target = -2;
	} else if (devp != NULL) {
		printf("WARNING: device not valid, trying to use default target...\n");
		scsibus = 0;
		target = 6;
		lun = 0;
	}
	if (be_verbose && scsidev != NULL) {
		printf("scsidev: '%s'\n", scsidev);
		if (devname[0] != '\0')
			printf("devname: '%s'\n", devname);
		printf("scsibus: %d target: %d lun: %d\n",
					scsibus, target, lun);
	}
	if (scsi_open() == 0)
		return -1;
	global.target = target;
	global.lun = lun;
	return 1;
}



#if defined (__linux__)
/*******************************************************************
 *
 *	cooked ioctl section
 *
 */
static struct cdrom_tochdr hdr;
static struct cdrom_tocentry entry[100];
static struct cdrom_read_audio arg;
static int err;

static subq_chnl *ReadSubQ_cooked __PR(( unsigned char sq_format, unsigned char track ));
static unsigned ReadToc_cooked __PR(( TOC *toc ));

/* read the table of contents (toc) via the ioctl interface */
static unsigned ReadToc_cooked ( toc )
	TOC *toc;
{
    unsigned i;
    unsigned tracks;

    err = ioctl( global.cooked_fd, CDROMSTOP, NULL );
    if ( err != 0 ) {
	/* error handling */
	if (err == -1) {
	    if (errno == EPERM)
		fprintf( stderr, "Please run this program setuid root.\n");
	    perror("cooked: STOP CDROM ");
	    exit( err );
	} else {
	    fprintf( stderr, "can't stop cdrom (error %d).\n", err );
	    exit( -1 );
	}
    }
    err = ioctl( global.cooked_fd, CDROMSTART, NULL );
    if ( err != 0 ) {
	/* error handling */
	if (err == -1) {
	    if (errno == EPERM)
		fprintf( stderr, "Please run this program setuid root.\n");
	    perror("cooked: START CDROM ");
	    exit( err );
	} else {
	    fprintf( stderr, "can't start cdrom (error %d).\n", err );
	    exit( -1 );
	}
    }
    /* get TocHeader to find out how many entries there are */
    err = ioctl( global.cooked_fd, CDROMREADTOCHDR, &hdr );
    if ( err != 0 ) {
	/* error handling */
	if (err == -1) {
	    if (errno == EPERM)
		fprintf( stderr, "Please run this program setuid root.\n");
	    perror("cooked: Read TOC ");
	    exit( err );
	} else {
	    fprintf( stderr, "can't get TocHeader (error %d).\n", err );
	    exit( -1 );
	}
    }
    /* get all TocEntries */
    for ( i = 0; i < hdr.cdth_trk1; i++ ) {
	entry[i].cdte_track = 1+i;
	entry[i].cdte_format = CDROM_LBA;
	err = ioctl( global.cooked_fd, CDROMREADTOCENTRY, &entry[i] );
	if ( err != 0 ) {
	    /* error handling */
	    fprintf( stderr, "can't get TocEntry #%d (error %d).\n", i+1, err );
	    exit( -1 );
	}
    }
    entry[i].cdte_track = CDROM_LEADOUT;
    entry[i].cdte_format = CDROM_LBA;
    err = ioctl( global.cooked_fd, CDROMREADTOCENTRY, &entry[i] );
    if ( err != 0 ) {
	/* error handling */
	fprintf( stderr, "can't get TocEntry LEADOUT (error %d).\n", err );
	exit( -1 );
    }
    tracks = hdr.cdth_trk1+1;
    for (i = 0; i < tracks; i++) {
        toc[i].bFlags = (entry[i].cdte_adr << 4) | (entry[i].cdte_ctrl & 0x0f);
        toc[i].bTrack = entry[i].cdte_track;
        toc[i].dwStartSector = entry[i].cdte_addr.lba;
    }
    return --tracks;           /* without lead-out */
}

static void trash_cache_cooked __PR((UINT4 *p, unsigned lSector, unsigned SectorBurstVal));

static void trash_cache_cooked(p, lSector, SectorBurstVal)
	UINT4 *p;
	unsigned lSector;
	unsigned SectorBurstVal;
{
      /* trash the cache */
      static struct cdrom_read_audio arg2;

      arg2.addr.lba = find_an_off_sector(lSector, SectorBurstVal);
      arg2.addr_format = CDROM_LBA;
      arg2.nframes = 3;
      arg2.buf = (unsigned char *) &p[0];
      ioctl(global.cooked_fd, CDROMREADAUDIO, &arg2);
}

static void ReadCdRom_cooked __PR(( UINT4 *p, unsigned lSector, unsigned SectorBurstVal));
/* read 'SectorBurst' adjacent sectors of audio sectors 
 * to Buffer '*p' beginning at sector 'lSector'
 */
static void ReadCdRom_cooked (p, lSector, SectorBurstVal )
	UINT4 *p;
	unsigned lSector;
	unsigned SectorBurstVal;
{
  int retry_count=0;
  static int nothing_read = 1;

/* read 2352 bytes audio data */
  arg.addr.lba = lSector;
  arg.addr_format = CDROM_LBA;
  arg.nframes = SectorBurstVal;
  arg.buf = (unsigned char *) &p[0];

  do {
    err = ioctl(global.cooked_fd, CDROMREADAUDIO, &arg);
    retry_count++;

    if (err) { 
      trash_cache_cooked(p, lSector, SectorBurstVal);
    }

  } while ((err) && (retry_count < 30));
  if (err != 0) {
      /* error handling */
      if (err == -1) {
	  if (nothing_read && (errno == EINVAL || errno == EIO))
	      fprintf( stderr, "Sorry, this driver and/or drive does not support cdda reading.\n");
	  perror("cooked: Read cdda ");
          fprintf(stderr, " sector %u + %u, buffer %p, %p + %x\n", lSector, SectorBurstVal, p, fill_buffer, global.shmsize);
      } else {
	  fprintf(stderr, "can't read frame #%d (error %d).\n", 
		  arg.addr.lba, err);
      }
  } else {
    nothing_read = 0;
  }
}

static int Play_at_cooked __PR(( unsigned int from_sector, unsigned int sectors));
static int Play_at_cooked( from_sector, sectors)
	unsigned int from_sector;
	unsigned int sectors;
{
	struct cdrom_msf cmsf;

	cmsf.cdmsf_min0 = (from_sector + 150) / (60*75);
	cmsf.cdmsf_sec0 = ((from_sector + 150) / 75) % 60;
	cmsf.cdmsf_frame0 = (from_sector + 150) % 75;
	cmsf.cdmsf_min1 = (from_sector + 150 + sectors) / (60*75);
	cmsf.cdmsf_sec1 = ((from_sector + 150 + sectors) / 75) % 60;
	cmsf.cdmsf_frame1 = (from_sector + 150 + sectors) % 75;

	ioctl( global.cooked_fd, CDROMSTART, 0 );
	return ioctl( global.cooked_fd, CDROMPLAYMSF, &cmsf ) ? 0 : -1; 
}

/* request sub-q-channel information. This function may cause confusion
 * for a drive, when called in the sampling process.
 */
static subq_chnl *ReadSubQ_cooked ( sq_format, track )
	unsigned char sq_format;
	unsigned char track;
{
    struct cdrom_subchnl sub_ch;

    switch (sq_format) {
      case GET_CATALOGNUMBER:
      if (!(err = ioctl(global.cooked_fd, CDROM_GET_MCN, (struct cdrom_mcn *) SubQbuffer))) {
          subq_chnl *SQp = (subq_chnl *) SubQbuffer;
	  subq_catalog *SQPp = (subq_catalog *) &SQp->data;

          movebytes(SQp, SQPp->media_catalog_number, sizeof (SQPp->media_catalog_number));
          SQPp->zero = 0;
          SQPp->mc_valid = 0x80;
      } else {
          return NULL;
      }
      break;
      case GET_POSITIONDATA:
      if (!(err = ioctl(global.cooked_fd, CDROMSUBCHNL, &sub_ch))) {
	  /* copy to SubQbuffer */
	  subq_chnl *SQp = (subq_chnl *) (SubQbuffer);
	  subq_position *SQPp = (subq_position *) &SQp->data;
	  SQp->audio_status 	= sub_ch.cdsc_audiostatus;
	  SQp->format 		= sub_ch.cdsc_format;
	  SQp->control_adr	= (sub_ch.cdsc_adr << 4) | (sub_ch.cdsc_ctrl & 0x0f);
	  SQp->track 		= sub_ch.cdsc_trk;
	  SQp->index 		= sub_ch.cdsc_ind;
	  SQPp->abs_min 	= sub_ch.cdsc_absaddr.msf.minute;
	  SQPp->abs_sec 	= sub_ch.cdsc_absaddr.msf.second;
	  SQPp->abs_frame 	= sub_ch.cdsc_absaddr.msf.frame;
	  SQPp->trel_min 	= sub_ch.cdsc_reladdr.msf.minute;
	  SQPp->trel_sec 	= sub_ch.cdsc_reladdr.msf.second;
	  SQPp->trel_frame 	= sub_ch.cdsc_reladdr.msf.frame;
      } else {
	  if (err == -1) {
	      if (errno == EPERM)
		  fprintf( stderr, "Please run this program setuid root.\n");
	      perror("cooked: Read subq ");
	      exit( err );
	  } else {
	      fprintf(stderr, "can't read sub q channel (error %d).\n", err);
	      exit( -1 );
	  }
      }
      break;
      default:
          return NULL;
    } /* switch */
  return (subq_chnl *)(SubQbuffer);
}

/* Speed control */
static void SpeedSelect_cooked __PR(( unsigned speed));
static void SpeedSelect_cooked( speed )
	unsigned speed;
{
#ifdef CDROM_SELECT_SPEED
    /* CAUTION!!!!! Non standard ioctl parameter types here!!!! */
    if (!(err = ioctl(global.cooked_fd, CDROM_SELECT_SPEED, speed))) {
    } else {
	if (err == -1) {
	    if (errno == EPERM)
		fprintf( stderr, "Please run this program setuid root.\n");
	    perror("cooked: Speed select ");
	    /*exit( err ); */
	} else {
	    fprintf(stderr, "can't set speed %d (error %d).\n", speed, err);
	    exit( -1 );
	}
    }
#else

    PRETEND_TO_USE(speed);
#endif
}

/* set function pointers to use the ioctl routines */
static void SetupCookedIoctl __PR(( char *pdev_name));
static void SetupCookedIoctl( pdev_name )
	char *pdev_name;
{
#if (HAVE_ST_RDEV == 1)
    struct stat statstruct;

    if (fstat(global.cooked_fd, &statstruct)) {
      fprintf(stderr, "cannot stat cd %d (%s)\n",global.cooked_fd, pdev_name);
      exit(1);
    }
    switch ((int)(statstruct.st_rdev >> 8L)) {
    case CDU31A_CDROM_MAJOR:	/* sony cdu-31a/33a */
        global.nsectors = 13;
        if (global.nsectors >= 14) {
	  global.overlap = 10;
	}
        break;
    case MATSUSHITA_CDROM_MAJOR:	/* sbpcd 1 */
    case MATSUSHITA_CDROM2_MAJOR:	/* sbpcd 2 */
    case MATSUSHITA_CDROM3_MAJOR:	/* sbpcd 3 */
    case MATSUSHITA_CDROM4_MAJOR:	/* sbpcd 4 */
        /* some are more compatible than others */
        global.nsectors = 13;
	break;
	default:
    }
    err = ioctl(global.cooked_fd, CDROMAUDIOBUFSIZ, global.nsectors);

    switch ((int)(statstruct.st_rdev >> 8L)) {
    case MATSUSHITA_CDROM_MAJOR:	/* sbpcd 1 */
    case MATSUSHITA_CDROM2_MAJOR:	/* sbpcd 2 */
    case MATSUSHITA_CDROM3_MAJOR:	/* sbpcd 3 */
    case MATSUSHITA_CDROM4_MAJOR:	/* sbpcd 4 */
      if (err == -1) {
        perror("ioctl(CDROMAUDIOBUFSIZ)");
      }
    }
#endif
    EnableCdda = (void (*)(int))Dummy;
    ReadCdRom = ReadCdRom_cooked;
    ReadCdRomData = (void (*)(unsigned char *, unsigned, unsigned )) ReadCdRom_cooked;
    ReadToc = ReadToc_cooked;
    ReadSubQ = ReadSubQ_cooked;
    SelectSpeed = SpeedSelect_cooked;
    Play_at = Play_at_cooked;
    trash_cache = trash_cache_cooked;
    ReadLastAudio = NULL;
}
#endif

/********************** General setup *******************************/

/* As the name implies, interfaces and devices are checked.  We also
   adjust nsectors, overlap, and interface for the first time here.
   Any unnecessary privileges (setuid, setgid) are also dropped here.
*/
static void Check_interface_for_device( statstruct, pdev_name)
	struct stat *statstruct;
	char *pdev_name;
{
    static const char *int_names[]={
       "generic_scsi", "cooked_ioctl"
    };


#if !defined (STAT_MACROS_BROKEN) || (STAT_MACROS_BROKEN != 1)
    if (!S_ISCHR(statstruct->st_mode) &&
	!S_ISBLK(statstruct->st_mode)) {
      fprintf(stderr, "%s is not a device\n",pdev_name);
      exit(1);
    }
#endif

#if defined (HAVE_ST_RDEV) && (HAVE_ST_RDEV == 1)
    switch ((int) (statstruct->st_rdev >> 8L)) {
#if defined (__linux__)
    case SCSI_GENERIC_MAJOR:	/* generic */
#else
    default:			/* ??? what is the proper value here */
#endif
#if !defined (STAT_MACROS_BROKEN) || (STAT_MACROS_BROKEN != 1)
       if (!S_ISCHR(statstruct->st_mode)) {
	 fprintf(stderr, "%s is not a char device\n",pdev_name);
	 exit(1);
       }
       if (interface != GENERIC_SCSI) {
	 if (!S_ISCHR(statstruct->st_mode)) {
	   fprintf(stderr, "%s is not a char device\n",pdev_name);
	   exit(1);
	 }
	    
	 fprintf(stderr, "wrong interface (%s) for this device (%s)\nset to generic_scsi\n",int_names[interface], pdev_name);
	 interface = GENERIC_SCSI;
       }
#endif
       break;
#if defined (__linux__)
    case SCSI_CDROM_MAJOR:     /* scsi cd */
    default:
	if (!S_ISBLK(statstruct->st_mode)) {
	    fprintf(stderr, "%s is not a block device\n",pdev_name);
	    exit(1);
	}
	if (interface != COOKED_IOCTL) {
	    fprintf(stderr, "cdrom device (%s) is not of type generic SCSI. "
		    "Setting interface to cooked_ioctl.\n", pdev_name);
	    interface = COOKED_IOCTL;
	}
	break;
#endif
    }
#endif
    if (global.overlap >= global.nsectors)
      global.overlap = global.nsectors-1;
}

/* open the cdrom device */
static int OpenCdRom ( pdev_name )
	char *pdev_name;
{
  int retval = 0;
  int have_named_device = 0;
  struct stat statstruct, fstatstruct;

  /*  The device (given by pdevname) can be:
      a. an SCSI device specified with a /dev/xxx name,
      b. an SCSI device specified with bus,target,lun numbers,
      c. a non-SCSI device such as ATAPI or proprietary CDROM devices.
   */
#ifdef __linux__
  have_named_device = strchr(pdev_name, ':') == NULL
			 && memcmp(pdev_name, "/dev/", 5) == 0;

  if (have_named_device) {
    if (stat(pdev_name, &statstruct)) {
      fprintf(stderr, "cannot stat device %s\n", pdev_name);
      exit(1);
    } else {
      Check_interface_for_device( &statstruct, pdev_name );
    }
  }
#endif

  if (interface == GENERIC_SCSI) {
      needroot(0);
      needgroup(0);
		/* device name, os_debug, verboseopen */
      retval = open_scsi(pdev_name, 0, 0);

      if (retval <= 0) {
        fprintf(stderr, "open(%s) in file %s, line %d: ",pdev_name, __FILE__, __LINE__);
        perror("open error for scsi device");
        dontneedgroup();
        dontneedroot();
#if defined (__linux__)
        fprintf(stderr, "On Linux make sure you have the generic SCSI driver installed.\n");
#endif
#if defined(sun) || defined(__sun) || defined(__sun__)
        fprintf(stderr, "On SunOS/Solaris make sure you have Joerg Schillings scg SCSI driver installed.\n");
#endif
        fprintf(stderr, "Probably you did not define your SCSI device.\n");
        fprintf(stderr, "Set the CDDA_DEVICE environment variable or use the -D option.\n");
        fprintf(stderr, "You can also define the default device in the Makefile.\n");
        exit(1);
      }
      scsi_settimeout(100);
      jes_verbose = global.scsi_verbose;
      dontneedgroup();
      dontneedroot();
      if (global.nsectors > (unsigned) scsi_bufsize(100*1024*1024)/CD_FRAMESIZE_RAW)
        global.nsectors = scsi_bufsize(100*1024*1024)/CD_FRAMESIZE_RAW;
      if (global.overlap >= global.nsectors)
        global.overlap = global.nsectors-1;

	init_scsibuf(global.nsectors*CD_FRAMESIZE_RAW);
  } else {
      needgroup(0);
      retval = open(pdev_name,O_RDONLY);
      dontneedgroup();

      if (retval < 0) {
        fprintf(stderr, "while opening %s :", pdev_name);
        perror("ioctl cdrom device open error: ");
        exit(1);
      }

      /* Do final security checks here */
      if (fstat(retval, &fstatstruct)) {
        fprintf(stderr, "Could not fstat %s (fd %d): ", pdev_name, retval);
        perror("ioctl fstat error: ");
        exit(1);
      }
      Check_interface_for_device( &fstatstruct, pdev_name );

      /* Watch for race conditions */
      if (have_named_device 
          && (fstatstruct.st_dev != statstruct.st_dev ||
              fstatstruct.st_ino != statstruct.st_ino)) {
         fprintf(stderr,"Race condition attempted in OpenCdRom.  Exiting now.\n");
         exit(1);
      }
  }
  return retval;
}
#endif /* SIM_CD */

/******************* Simulation interface *****************/
#if	defined SIM_CD
#include "toc.h"
static unsigned long sim_pos=0;

/* read 'SectorBurst' adjacent sectors of audio sectors 
 * to Buffer '*p' beginning at sector 'lSector'
 */
static void ReadCdRom_sim __PR(( UINT4 *p, unsigned lSector, unsigned SectorBurstVal));
static void ReadCdRom_sim ( p, lSector, SectorBurstVal )
	UINT4 *p;
	unsigned lSector;
	unsigned SectorBurstVal;
{
  unsigned int loop=0;
  short *q = (short *) p;
  int joffset = 0;

  if (lSector > g_toc[cdtracks].dwStartSector || lSector + SectorBurstVal > g_toc[cdtracks].dwStartSector + 1) {
    fprintf(stderr, "Read request out of bounds: %u - %u (%d - %d allowed)\n",
	lSector, lSector + SectorBurstVal, 0, g_toc[cdtracks].dwStartSector);
  }
#if 0
  /* jitter with a probability of jprob */
  if (random() <= jprob) {
    /* jitter up to jmax samples */
    joffset = random();
  }
#endif

#ifdef DEBUG_SHM
  fprintf(stderr, ", last_b = %p\n", *last_buffer);
#endif
  for (loop = lSector*CD_FRAMESAMPLES + joffset; 
       loop < (lSector+SectorBurstVal)*CD_FRAMESAMPLES + joffset; 
       loop++) {
    *q++ = loop;
    *q++ = ~loop;
  }
#ifdef DEBUG_SHM
  fprintf(stderr, "sim wrote from %p upto %p - 4 (%d), last_b = %p\n",
          p, q, SectorBurstVal*CD_FRAMESAMPLES, *last_buffer);
#endif
  sim_pos = (lSector+SectorBurstVal)*CD_FRAMESAMPLES + joffset; 
}

static int Play_at_sim __PR(( unsigned int from_sector, unsigned int sectors));
static int Play_at_sim( from_sector, sectors)
	unsigned int from_sector;
	unsigned int sectors;
{
  sim_pos = from_sector*CD_FRAMESAMPLES; 
  return 0;
}

static unsigned sim_indices;


/* read the table of contents (toc) via the ioctl interface */
static unsigned ReadToc_sim __PR(( TOC *toc));
static unsigned ReadToc_sim ( toc )
	TOC *toc;
{
    unsigned int scenario;
    int scen[12][3] = { 
      {1,1,5000}, 
      {1,2,5000}, 
      {1,99,150*99}, 
      {2,1,5000}, 
      {2,2,5000}, 
      {2,99,150*99},
      {5,1,5000}, 
      {5,2,5000}, 
      {5,99,150*99}, 
      {99,1,1000}, 
      {99,2,1000}, 
      {99,99,150*99}, 
    };
    unsigned int i;
    unsigned trcks;

    fprintf(stderr, "select one of the following TOCs\n"
	    "0 :  1 track  with  1 index\n"
	    "1 :  1 track  with  2 indices\n"
	    "2 :  1 track  with 99 indices\n"
	    "3 :  2 tracks with  1 index each\n"
	    "4 :  2 tracks with  2 indices each\n"
	    "5 :  2 tracks with 99 indices each\n"
	    "6 :  5 tracks with  1 index each\n"
	    "7 :  5 tracks with  2 indices each\n"
	    "8 :  5 tracks with 99 indices each\n"
	    "9 : 99 tracks with  1 index each\n"
	    "10: 99 tracks with  2 indices each\n"
	    "11: 99 tracks with 99 indices each\n"
	    );

    do {
      scanf("%u", &scenario);
    } while (scenario > sizeof(scen)/2/sizeof(int));

    /* build table of contents */

    trcks = scen[scenario][0] + 1;
    sim_indices = scen[scenario][1];

    for (i = 0; i < trcks; i++) {
        toc[i].bFlags = 0x1b;
        toc[i].bTrack = i + 1;
        toc[i].dwStartSector = i * scen[scenario][2];
    }
    toc[i].bTrack = 0xaa;
    return --trcks;           /* without lead-out */
}


static subq_chnl *ReadSubQ_sim __PR(( unsigned char sq_format, unsigned char track ));
/* request sub-q-channel information. This function may cause confusion
 * for a drive, when called in the sampling process.
 */
static subq_chnl *ReadSubQ_sim ( sq_format, track )
	unsigned char sq_format;
	unsigned char track;
{
    subq_chnl *SQp = (subq_chnl *) (SubQbuffer);
    subq_position *SQPp = (subq_position *) &SQp->data;
    unsigned long sim_pos1;
    unsigned long sim_pos2;

    if ( sq_format != GET_POSITIONDATA ) return NULL;  /* not supported by sim */

    /* simulate CDROMSUBCHNL ioctl */

    /* copy to SubQbuffer */
    SQp->audio_status 	= 0;
    SQp->format 	= 0xff;
    SQp->control_adr	= 0xff;
    sim_pos1 = sim_pos/CD_FRAMESAMPLES;
    sim_pos2 = sim_pos1 % 150;
    SQp->track 		= (sim_pos1 / 5000) + 1;
    SQp->index 		= ((sim_pos1 / 150) % sim_indices) + 1;
    sim_pos1 += 150;
    SQPp->abs_min 	= sim_pos1 / (75*60);
    SQPp->abs_sec 	= (sim_pos1 / 75) % 60;
    SQPp->abs_frame 	= sim_pos1 % 75;
    SQPp->trel_min 	= sim_pos2 / (75*60);
    SQPp->trel_sec 	= (sim_pos2 / 75) % 60;
    SQPp->trel_frame 	= sim_pos2 % 75;

    return (subq_chnl *)(SubQbuffer);
}

static void SelectSpeed_sim __PR(( unsigned sp));
static void SelectSpeed_sim(sp)
	unsigned sp;
{
}

static void trash_cache_sim __PR((UINT4 *p, unsigned lSector, unsigned SectorBurstVal));

static void trash_cache_sim(p, lSector, SectorBurstVal)
	UINT4 *p;
	unsigned lSector;
	unsigned SectorBurstVal;
{
	PRETEND_TO_USE(p);
	PRETEND_TO_USE(lSector);
	PRETEND_TO_USE(SectorBurstVal);
}

static void SetupSimCd __PR((void));

static void SetupSimCd()
{
    EnableCdda =  (void (*) __PR((int)))Dummy;
    ReadCdRom = ReadCdRom_sim;
    ReadCdRomData = (void (*) __PR((unsigned char *, unsigned, unsigned ))) ReadCdRom_sim;
    ReadToc = ReadToc_sim;
    ReadSubQ = ReadSubQ_sim;
    ReadLastAudio = NULL;
    SelectSpeed = SelectSpeed_sim;
    Play_at = Play_at_sim;
    trash_cache = trash_cache_sim;
 
#if 0
    fprintf( stderr, "Please enter values for 'global.littleendian',  'overlap' :");
    scanf("%d,%d", &global.littleendian, &global.overlap); 
#else
    global.littleendian = 1;
    global.overlap = min(global.nsectors-1, 1);
#endif
}

#endif /* def SIM_CD */

/* perform initialization depending on the interface used. */
void SetupInterface( )
{
#define SETSIGHAND(SIG, SIGNAME) if (signal(SIG, exit) == SIG_ERR) \
	{ fprintf(stderr, "cannot set signal %s handler\n", SIGNAME); exit(1); }
    SETSIGHAND(SIGINT, "SIGINT")
    SETSIGHAND(SIGQUIT, "SIGQUIT")
    SETSIGHAND(SIGTERM, "SIGTERM")
    SETSIGHAND(SIGHUP, "SIGHUP")
    SETSIGHAND(SIGPIPE, "SIGPIPE")
    SETSIGHAND(SIGTRAP, "SIGTRAP")
#if !defined(SIGIOT)
#define SIGIOT SIGIO
#endif
    SETSIGHAND(SIGIOT, "SIGIOT")


#if	defined SIM_CD
    global.nsectors = 75;
    fprintf( stderr, "SIMULATION MODE !!!!!!!!!!!\n");
#else
    /* ensure interface is setup correctly */
    global.cooked_fd = OpenCdRom ( global.dev_name );
#endif

    /* Value of 'nsectors' must be defined here */
    assert(global.nsectors > 0);

    global.shmsize = sizeof(struct ringbuffheader) + 
                    (global.nsectors-1)*CD_FRAMESIZE_RAW +
                    (global.buffers-1)*
                    (sizeof(myringbuff)+(global.nsectors-1)*CD_FRAMESIZE_RAW);

#if	defined (HAVE_FORK_AND_SHAREDMEM)
#if	defined(HAVE_SMMAP) || defined(HAVE_USGSHM)
    fill_buffer = request_shm_sem(global.shmsize, (unsigned char **)&fill_buffer);
    if (fill_buffer == NULL) {
#else /* have shared memory */
    if (1) {
#endif
	fprintf( stderr, "no shared memory available!\n");
	exit(2);
    }
#else /* do not have fork() and shared memory */
    fill_buffer = malloc(global.shmsize);
    if (fill_buffer == NULL) {
	fprintf( stderr, "no buffer memory available!\n");
	exit(2);
    }
#endif

    if (global.verbose != 0)
   	 fprintf(stderr,
   		 "%u bytes buffer memory requested, %d buffers, %d sectors\n",
   		 global.shmsize, global.buffers, global.nsectors);
    last_buffer = fill_buffer + 1;
    set_total_buffers(global.buffers, sem_id);

    /* request one sector for table of contents */
    bufferTOC = (unsigned char *) malloc( CD_FRAMESIZE );      /* assumes sufficient aligned addresses */
    /* SubQchannel buffer */
    SubQbuffer = (subq_chnl *) malloc( 48 );               /* assumes sufficient aligned addresses */
    cmd = (unsigned char *) malloc( 18 );                      /* assumes sufficient aligned addresses */
    if ( !bufferTOC || !SubQbuffer || !cmd ) {
       fprintf( stderr, "Too low on memory. Giving up.\n");
       exit(2);
    }

#if	defined SIM_CD
    SetupSimCd();
#else
    adjust_ssize = 1;
    /* if drive is of type scsi, get vendor name */
    if (interface == GENERIC_SCSI) {
        unsigned sector_size;

	SetupSCSI();
        sector_size = get_orig_sectorsize(&orgmode4, &orgmode10, &orgmode11);
	if (!SCSI_emulated_ATAPI_on()) {
          if ( sector_size != 2048 && set_sectorsize(2048) ) {
	    fprintf( stderr, "Could not change sector size from %d to 2048\n", sector_size );
	    adjust_ssize = 2048 / sector_size;
          }
        } else {
          sector_size = 2048;
          adjust_ssize = 1;
        }

	/* get cache setting */

	/* set cache to zero */

#if defined (__linux__)
    } else {
	SetupCookedIoctl( global.dev_name );
#endif
    }
#endif	/* if def SIM_CD */
}
